#include "gfx/gp32splash.h"
#include "gfx/gp32menu.h"

int game_num_avail=0;		/* Number of available games */
int game_index=0;		/* Absolute index of selected game */
int last_game_selected=0;	/* Dynamic Index of last selected game */
GPDRAWSURFACE gpDraw[2];	/* Video Buffers */
int nflip;			/* Video buffer index */

struct node {
      char* stritem;
      struct node* prev;
      struct node* next;
};
char romName_argument[128];
struct node* available_games_list;
char* gamelist_cfg = "gp:\\gpmm\\mamegp32\\cfg\\gamelist.cfg\0";

struct KeySettings {
	int JOY_LEFT;
	int JOY_RIGHT;
	int JOY_UP;
	int JOY_DOWN;
	int JOY_FIRE1;
	int JOY_FIRE2;
	int JOY_FIRE3;
	int JOY_FIRE4;
	int JOY_FIRE5;
	int JOY_FIRE6;
	int JOY_FIRE7;
	int JOY_FIRE8;
	int JOY_FIRE9;
	int JOY_FIRE10;
        int JOY_FIRE1_AUTO;
        int JOY_FIRE2_AUTO;
        int JOY_FIRE3_AUTO;
};
static struct KeySettings keySettings;
struct KeySettings *key_settings = &keySettings;
#define KEY_FILENAME "gp:\\gpmm\\mamegp32\\key\\%s.key\0"
#define KEY_STRUCTURE "LEFT=%d;RIGHT=%d;UP=%d;DOWN=%d;FIRE1=%d;FIRE2=%d;FIRE3=%d;FIRE4=%d;FIRE5=%d;FIRE6=%d;FIRE7=%d;FIRE8=%d;FIRE9=%d;FIRE10=%d;AUTO1=%d;AUTO2=%d;AUTO3=%d;\0"
#define JOY_STRUCTURE "x_sensitivity=%d;y_sensitivity=%d;x_reversed=%d;y_reversed=%d;\0"
#define JOY_FILENAME "gp:\\gpmm\\mamegp32\\joy\\%s.joy\0"

#define RGBA(R,G,B,A) (((R&0xF8)<<8)|((G&0xF8)<<3)|((B&0xF8)>>2)|A) /* Converts 8bit rgb values to a GP32 palette value */
static unsigned long *gpPalette=(unsigned long *)0x14A00400; /* Pointer to Palette */ 

/* Game List Text Out */
void gp32_gamelist_text_out(int x, int y, char *eltexto) {
	char texto[33];
	gm_strncpy(texto,eltexto,32);
	if (texto[0]!='-')
		GpTextOut( NULL, &gpDraw[nflip],x+1,y+1,texto, 0 /* 0xff */ );
	GpTextOut( NULL, &gpDraw[nflip],x,y,texto, 100 /* 0xff */ );
}

/* Debug information function*/
void gp32_text_out(int x, int y, char *texto) {
	GpTextOut( NULL, &gpDraw[nflip],x,y,texto, 0x01 /* 0xff */ );
}

/* Debug information function*/
void gp32_text_out_int(int x, int y, int entero) {
	char cad[50];
	sprintf(cad,"%d\n",entero);
	gp32_text_out(x,y,cad);
}

int log=0;

void clear_screen(void)
{
	/* Clean screen*/
	GpRectFill( NULL, &gpDraw[0],0,0,gpDraw[0].buf_w,gpDraw[0].buf_h,0x0 );
	GpRectFill( NULL, &gpDraw[1],0,0,gpDraw[1].buf_w,gpDraw[1].buf_h,0x0 );
}

void draw_border(void)
{
	GpBitBlt( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,(unsigned char*)gp32menu,0,0,gp32menu_width,gp32menu_height);
}

/* Set Debug Log start at row 0 */
void gp32_gamelist_zero(void) {
	log=0;
}

/* Debug Log information funcion */
void gp32_text_log(char *texto) {
	if (log==0)
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );
		
	gp32_text_out(0,log,texto);
	log+=15;
	if(log>239) log=0;
}

/* Debug Log information funcion */
void gp32_text_log_int(int entero) {
	if (log==0)
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );
	
	gp32_text_out_int(0,log,entero);
	log+=15;
	if(log>239) log=0;
}

int gp32_fexists ( const char * filename )
  {
    GPFILEATTR attr;
    
    if (GpFileAttr(filename, &attr)!=SM_OK)
      return 0;
    
    return 1;
  }

void game_list_view(int *pos) {

	int i;
	int view_pos;
	int aux_pos=0;
	int screen_y = 38;
	int screen_x = 40;
	int x_pos = 255;
	char game_pos[16];

	draw_border();

	if(game_num_avail==0) {
		*pos=-1;
	}

	/* Check Limits */
	if (*pos>(game_num_avail-1))
		*pos=*pos-(game_num_avail-1);
	if (*pos<0)
		*pos=(game_num_avail-1)+*pos;
					   
	/* Set View Pos */
	if (*pos<5) {
		view_pos=0;
	} else {
		if (*pos>game_num_avail-7) {
			view_pos=game_num_avail-11;
			view_pos=(view_pos<0?0:view_pos);
		} else {
			view_pos=*pos-5;
		}
	}

	/* Show List */
	for (i=0;i<NUMGAMES;i++) {
		if (drivers[i].available==1) {
			if (aux_pos>=view_pos && aux_pos<=view_pos+10) {
				gp32_gamelist_text_out( screen_x, screen_y, (char *)drivers[i].description );
				if (aux_pos==*pos) {
					gp32_gamelist_text_out( screen_x-10, screen_y,">" );
					gp32_gamelist_text_out( screen_x-13, screen_y,"-" );
				}
				screen_y+=15;
			}
			aux_pos++;
		}
	}

	/* Show position in list */
	sprintf(game_pos,"%d/%d",*pos+1,game_num_avail);
	x_pos -= *pos+1 >= 10 ? 8:0;
	x_pos -= *pos+1 >= 100 ? 8:0;
	GpTextOut( NULL, &gpDraw[nflip],x_pos,22,game_pos, 254 );
}

int game_list_select (int index) {
	int i;
	int aux_pos=0;
	for (i=0;i<NUMGAMES;i++) 
	{
		if (drivers[i].available==1) 
		{
			if(aux_pos==index) 
			{
				return i;
			}
			aux_pos++;
		}
	}
	return -1;
}

void show_selected_game(void)
{
	char text[64];
	game_index = game_list_select(last_game_selected);
	gm_sprintf(text, "Selected Game: %s\0", drivers[game_index].name);
	gp32_gamelist_text_out(40,38,text);
	gp32_gamelist_text_out(40,38+12,drivers[game_index].description);
}

void mark_game_as_available(char *romname)
{
	int i;
	for (i=0; i<NUMGAMES; i++) {
		if( strcmp(romname, drivers[i].name) == 0) {
			if( gm_lstrlen(romName_argument) && strcmp(romName_argument, romname) == 0) {
				/* Make the passed in romname the selected item in the list */
				last_game_selected = game_num_avail;
			}

			/* Add this romname to the list of available games */
			drivers[i].available=1;
			game_num_avail++;
			break;
		}
	}
}

void append(struct node **s, char* stritem) 
{ 
      struct node *r, *q = *s ; 
      if (*s == NULL) 
      { 
            /* 1st time in, create a new list */
            *s = gm_zi_malloc(sizeof(struct node)) ; 
            (*s) -> prev = NULL; 
            (*s) -> stritem = stritem; 
            (*s) -> next = NULL ; 
      } 
      else 
      { 
            /* Find the end of the list */
            while (q -> next != NULL)
            {
                  q = q -> next;
            } 
 
            /* add it at the end */
            r = gm_zi_malloc(sizeof(struct node)); 
            r -> stritem = stritem; 
            r -> next = NULL; 
            r -> prev = q; 
            q -> next = r; 
      } 
}

void free_memory(void)
{
	struct node *thisNode = available_games_list;
	struct node *nextNode;

	while(thisNode != NULL)
	{
        nextNode = thisNode->next;
		if(thisNode->stritem != NULL){
			gm_free(thisNode->stritem);
		}
		gm_free(thisNode);
		thisNode = nextNode;
	}
}

struct node* processGameList(char* gameListString)
{
      int i, j, k;
      char* wrd;
      int gameStartPos = 0;
      int gameCount = 0;
      int lastCharWasEOL = 0;
      struct node* gameList;
      gameList = NULL;

      for (i = 0; i < strlen(gameListString); i++)
      {
            char ch = gameListString[i];
            if (ch == '\n' || ch == '\r')
            {
                  if (lastCharWasEOL == 1)
                  {
                        gameStartPos++;
                        continue;
                  }
                  lastCharWasEOL = 1;
                  /* we've hit a cr/nl char, so this game is complete */
                  wrd = gm_calloc((i - gameStartPos) + 1, sizeof(char));
                  k = 0;
                  for (j = gameStartPos; j < i; j++)
                  {
                        wrd[k] = gameListString[j];
                        k++;
                  }
                  wrd[k] = '\0';
                  append(&gameList, wrd);
                  gameStartPos = i + 1;
                  gameCount++;
            }
            else
            {
                  lastCharWasEOL = 0;
            }
      }
      return gameList;
}

void load_game_list(void)
{
	F_HANDLE fileHandle;
	ERR_CODE errCode;
	unsigned long dataSize;
	char *dataFromFile;
	int i;

	/* Reset all the data */
	last_game_selected = 0;
	game_num_avail = 0;
	for (i=0;i<NUMGAMES;i++)
	{
		drivers[i].available=0;
	}

	if( GpFileOpen( gamelist_cfg, OPEN_R, &fileHandle )==SM_OK )
	{
		GpFileGetSize ( gamelist_cfg, &dataSize );
		dataFromFile = gm_zi_malloc(dataSize);
		errCode = GpFileRead( fileHandle, dataFromFile, dataSize, NULL);
		if( errCode==SM_OK )
		{
			struct node* game_list = processGameList(dataFromFile);
			available_games_list = game_list;
            
			while( game_list != NULL )
			{
				if( game_list->stritem != NULL )
				{
					mark_game_as_available(game_list->stritem);
				}
				game_list = game_list->next;
			}
		}
		gm_free(dataFromFile);
		GpFileClose( fileHandle );
	}
}

void update_game_list(int clean_screen)
{
	F_HANDLE fileHandle;
	ERR_CODE errCode;
	int i = 0;
	char save_line[256];
	char* save_line_format = "%s\r\n";
	char romZip[128];
	char romDir[128];

	if (clean_screen) {
		clear_screen();
		GpTextOut( NULL, &gpDraw[nflip],15,110,"UPDATING THE LIST OF AVAILABLE GAMES", 254 );
	} else {
		GpTextOut( NULL, &gpDraw[nflip],6,6,"Updating the list of available games...", 0 );
		GpTextOut( NULL, &gpDraw[nflip],5,5,"Updating the list of available games...", 254 );
	}

	/* Reset data when the available game list is updated */
	strcpy(romName_argument,"");
	free_memory();

	if (GpFileCreate(gamelist_cfg, ALWAYS_CREATE, &fileHandle )==SM_OK) 
	{
		GpFileClose( fileHandle );
		if (GpFileOpen(gamelist_cfg, OPEN_W, &fileHandle )==SM_OK)
		{
			for (i=0;i<NUMGAMES;i++)
			{
				/* Search for game */
				sprintf(romZip,"gp:\\gpmm\\mamegp32\\roms\\%s.zip\0",drivers[i].name);
				sprintf(romDir,"gp:\\gpmm\\mamegp32\\roms\\%s\0",drivers[i].name);
				if (gp32_fexists(romZip) || gp32_fexists(romDir)) {
					sprintf(save_line, save_line_format, drivers[i].name);
					GpFileWrite(fileHandle, save_line, (gm_lstrlen(save_line)));
				}
			}

			GpFileClose( fileHandle );
			GpFATUpdate("GP://");
		}
	}

	/* Now reload this list of games */
	load_game_list();
}

/* Wait millisecs */
void Delay( int millisec )
{
	int nTick;
	nTick=GpTickCountGet();
	while( (GpTickCountGet()-nTick)<millisec );
}

int confirm_delete(char* filename, char* message)
{
	int ExKey, y_Pos=38;

	if(gp32_fexists(filename))
	{
		clear_screen();
		draw_border();
		show_selected_game();

		gp32_gamelist_text_out(40,y_Pos+50,"Are you sure you want to");
		gp32_gamelist_text_out(40,y_Pos+65,message);
		gp32_gamelist_text_out(40,y_Pos+155,"Press A to Confirm, B to Cancel");

		while(1)
		{
			while(GpKeyGet()!=GPC_VK_NONE) { }
			while((ExKey=GpKeyGet()) == GPC_VK_NONE) { }

			if(ExKey==GPC_VK_FA)
			{
				GpFileRemove(filename);
				GpFATUpdate("GP://");
				clear_screen();
				draw_border();
				return 1;
			}
			else if(ExKey==GPC_VK_FB)
			{
				/* Return to options */
				break;
			}
		}
	}

	clear_screen();
	draw_border();
	return 0;
}

void load_default_keys(void)
{
	key_settings->JOY_LEFT		= GPC_VK_LEFT;
	key_settings->JOY_RIGHT		= GPC_VK_RIGHT;
	key_settings->JOY_UP		= GPC_VK_UP;
	key_settings->JOY_DOWN		= GPC_VK_DOWN;
	key_settings->JOY_FIRE1		= GPC_VK_FA;
	key_settings->JOY_FIRE2		= GPC_VK_FB;
	key_settings->JOY_FIRE3		= GPC_VK_SELECT;
	key_settings->JOY_FIRE4		= GPC_VK_FL;
	key_settings->JOY_FIRE5		= GPC_VK_START;
	key_settings->JOY_FIRE6		= GPC_VK_FR;
	key_settings->JOY_FIRE7		= GPC_VK_SELECT;
	key_settings->JOY_FIRE8		= GPC_VK_FL;
	key_settings->JOY_FIRE9		= GPC_VK_START;
	key_settings->JOY_FIRE10	= GPC_VK_FR;
        key_settings->JOY_FIRE1_AUTO	= 0;
        key_settings->JOY_FIRE2_AUTO	= 0;
        key_settings->JOY_FIRE3_AUTO	= 0;
}

void load_keysettings(char* game_name)
{
	F_HANDLE fileHandle;
	ERR_CODE errCode;
	unsigned long dataSize;
	char *dataFromFile;
	char filename[64];

	/* Load default keys, used if .key file is missing */
	load_default_keys();
	sprintf(filename, KEY_FILENAME, game_name);

	if( GpFileOpen( filename, OPEN_R, &fileHandle )==SM_OK )
	{
		GpFileGetSize ( filename, &dataSize );
		dataFromFile = gm_zi_malloc(dataSize);
		errCode = GpFileRead( fileHandle, dataFromFile, dataSize, NULL);
		if( errCode==SM_OK )
		{
			sscanf( dataFromFile, KEY_STRUCTURE, &key_settings->JOY_LEFT, &key_settings->JOY_RIGHT, &key_settings->JOY_UP, &key_settings->JOY_DOWN, &key_settings->JOY_FIRE1, &key_settings->JOY_FIRE2, &key_settings->JOY_FIRE3, &key_settings->JOY_FIRE4, &key_settings->JOY_FIRE5, &key_settings->JOY_FIRE6, &key_settings->JOY_FIRE7, &key_settings->JOY_FIRE8, &key_settings->JOY_FIRE9, &key_settings->JOY_FIRE10, &key_settings->JOY_FIRE1_AUTO, &key_settings->JOY_FIRE2_AUTO, &key_settings->JOY_FIRE3_AUTO );
		}

		gm_free(dataFromFile);
		GpFileClose( fileHandle );
	}
}

int get_key_num(int currentKey, int getNextKey)
{
	switch(currentKey)
	{
	case GPC_VK_LEFT:
		return getNextKey ? GPC_VK_RIGHT : GPC_VK_FR;
	case GPC_VK_RIGHT:
		return getNextKey ? GPC_VK_UP : GPC_VK_LEFT;
	case GPC_VK_UP:
		return getNextKey ? GPC_VK_DOWN : GPC_VK_RIGHT;
	case GPC_VK_DOWN:
		return getNextKey ? GPC_VK_FA : GPC_VK_UP;
	case GPC_VK_FA:
		return getNextKey ? GPC_VK_FB : GPC_VK_DOWN;
	case GPC_VK_FB:
		return getNextKey ? GPC_VK_SELECT : GPC_VK_FA;
	case GPC_VK_SELECT:
		return getNextKey ? GPC_VK_START : GPC_VK_FB;
	case GPC_VK_START:
		return getNextKey ? GPC_VK_FL : GPC_VK_SELECT;
	case GPC_VK_FL:
		return getNextKey ? GPC_VK_FR : GPC_VK_START;
	case GPC_VK_FR:
		return getNextKey ? GPC_VK_LEFT : GPC_VK_FL;
	}

	/* Should never happen */
	return currentKey;
}

void get_key_name(char* outText, char* formatText, int key_num)
{
	switch(key_num)
	{
	case GPC_VK_LEFT:
		gm_sprintf(outText, formatText, "Joystick Left");
		break;
	case GPC_VK_RIGHT:
		gm_sprintf(outText, formatText, "Joystick Right");
		break;
	case GPC_VK_UP:
		gm_sprintf(outText, formatText, "Joystick Up");
		break;
	case GPC_VK_DOWN:
		gm_sprintf(outText, formatText, "Joystick Down");
		break;
	case GPC_VK_FA:
		gm_sprintf(outText, formatText, "A Button");
		break;
	case GPC_VK_FB:
		gm_sprintf(outText, formatText, "B Button");
		break;
	case GPC_VK_SELECT:
		gm_sprintf(outText, formatText, "Select Button");
		break;
	case GPC_VK_START:
		gm_sprintf(outText, formatText, "Start Button");
		break;
	case GPC_VK_FL:
		gm_sprintf(outText, formatText, "L Button");
		break;
	case GPC_VK_FR:
		gm_sprintf(outText, formatText, "R Button");
		break;
	}
}

void configure_keys(void)
{
	int ExKey, selected_option = 0, x_Pos = 40, y_Pos = 38+30, gaps = 0;
	int x_sensitivity = 0, y_sensitivity = 0, x_reversed = 0, y_reversed = 0;
	int options_count = 14;
	unsigned long dataSize;
	F_HANDLE fileHandle;
	char filename[128];
	char text[256];
	char* options[] = 
					{	"LEFT     =   %s",
						"RIGHT    =   %s",
						"UP       =   %s",
						"DOWN     =   %s",
						"FIRE1    =   %s",
						"FIRE2    =   %s",
						"FIRE3    =   %s",
						"FIRE4    =   %s",
						"FIRE5    =   %s",
						"FIRE6    =   %s",
						"FIRE7    =   %s",
						"FIRE8    =   %s",
						"FIRE9    =   %s",
						"FIRE10   =   %s" };

	clear_screen();

	game_index=game_list_select(last_game_selected);
	load_keysettings(drivers[game_index].name);

	while(1)
	{
		nflip=(nflip?0:1);
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );

		/* Draw the background */
		draw_border();

		show_selected_game();

		if(selected_option < 7)
		{
			get_key_name(text, options[0], key_settings->JOY_LEFT);
			gp32_gamelist_text_out(x_Pos,y_Pos,text);
			get_key_name(text, options[1], key_settings->JOY_RIGHT);
			gp32_gamelist_text_out(x_Pos,y_Pos+15,text);
			get_key_name(text, options[2], key_settings->JOY_UP);
			gp32_gamelist_text_out(x_Pos,y_Pos+30,text);
			get_key_name(text, options[3], key_settings->JOY_DOWN);
			gp32_gamelist_text_out(x_Pos,y_Pos+45,text);

			get_key_name(text, options[4], key_settings->JOY_FIRE1);
			if (key_settings->JOY_FIRE1_AUTO) gm_strcat(text, " (Auto)");
			gp32_gamelist_text_out(x_Pos,y_Pos+60,text);

			get_key_name(text, options[5], key_settings->JOY_FIRE2);
			if (key_settings->JOY_FIRE2_AUTO) gm_strcat(text, " (Auto)");
			gp32_gamelist_text_out(x_Pos,y_Pos+75,text);

			get_key_name(text, options[6], key_settings->JOY_FIRE3);
			if (key_settings->JOY_FIRE3_AUTO) gm_strcat(text, " (Auto)");
			gp32_gamelist_text_out(x_Pos,y_Pos+90,text);
		}
		else
		{
			get_key_name(text, options[7], key_settings->JOY_FIRE4);
			gp32_gamelist_text_out(x_Pos,y_Pos,text);
			get_key_name(text, options[8], key_settings->JOY_FIRE5);
			gp32_gamelist_text_out(x_Pos,y_Pos+15,text);
			get_key_name(text, options[9], key_settings->JOY_FIRE6);
			gp32_gamelist_text_out(x_Pos,y_Pos+30,text);
			get_key_name(text, options[10], key_settings->JOY_FIRE7);
			gp32_gamelist_text_out(x_Pos,y_Pos+45,text);
			get_key_name(text, options[11], key_settings->JOY_FIRE8);
			gp32_gamelist_text_out(x_Pos,y_Pos+60,text);
			get_key_name(text, options[12], key_settings->JOY_FIRE9);
			gp32_gamelist_text_out(x_Pos,y_Pos+75,text);
			get_key_name(text, options[13], key_settings->JOY_FIRE10);
			gp32_gamelist_text_out(x_Pos,y_Pos+90,text);
		}

		gp32_gamelist_text_out(x_Pos,y_Pos+110,"Press Select to load defaults");
		gp32_gamelist_text_out(x_Pos,y_Pos+125,"A to Save Changes, B to return");

		gaps = 0;
		if(selected_option >= 7) {
			gaps = -7*15;
		}

		/* Show currently selected item */
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-10,y_Pos+gaps+(selected_option*15),">", 254 );
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-13,y_Pos+gaps+(selected_option*15),"-", 254 );

		GpSurfaceSet( &gpDraw[nflip] );
		while(GpKeyGet()!=GPC_VK_NONE) { }
		while((ExKey=GpKeyGet()) == GPC_VK_NONE) { }

		GpTextOut( NULL, &gpDraw[nflip],x_Pos-10,y_Pos+gaps+(selected_option*15),">", 0 );
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-13,y_Pos+gaps+(selected_option*15),"-", 0 );

		if(ExKey==GPC_VK_DOWN){
			selected_option++;
			if(selected_option == 7)
			{
				/* Switch to next set of 7 options */
				clear_screen();
				draw_border();
			}
			else if(selected_option>options_count-1)
			{
				clear_screen();
				draw_border();
				selected_option = 0;
			}
		}
		else if(ExKey==GPC_VK_UP){
			selected_option--;
			if(selected_option == 6)
			{
				/* Switch to previous set of 7 options */
				clear_screen();
				draw_border();
			}
			else if(selected_option<0)
			{
				clear_screen();
				draw_border();
				selected_option = options_count - 1;
			}
		}
		else if(ExKey==GPC_VK_RIGHT || ExKey==GPC_VK_LEFT)
		{
			switch(selected_option)
			{
			case 0:
				get_key_name(text, options[13], key_settings->JOY_LEFT);
				key_settings->JOY_LEFT		= get_key_num(key_settings->JOY_LEFT, ExKey==GPC_VK_RIGHT ? 1 : 0);
				break;
			case 1:
				get_key_name(text, options[13], key_settings->JOY_RIGHT);
				key_settings->JOY_RIGHT		= get_key_num(key_settings->JOY_RIGHT, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 2:
				get_key_name(text, options[13], key_settings->JOY_UP);
				key_settings->JOY_UP		= get_key_num(key_settings->JOY_UP, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 3:
				get_key_name(text, options[13], key_settings->JOY_DOWN);
				key_settings->JOY_DOWN		= get_key_num(key_settings->JOY_DOWN, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 4:
				get_key_name(text, options[13], key_settings->JOY_FIRE1);
				key_settings->JOY_FIRE1		= get_key_num(key_settings->JOY_FIRE1, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 5:
				get_key_name(text, options[13], key_settings->JOY_FIRE2);
				key_settings->JOY_FIRE2		= get_key_num(key_settings->JOY_FIRE2, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 6:
				get_key_name(text, options[13], key_settings->JOY_FIRE3);
				key_settings->JOY_FIRE3		= get_key_num(key_settings->JOY_FIRE3, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 7:
				get_key_name(text, options[13], key_settings->JOY_FIRE4);
				key_settings->JOY_FIRE4		= get_key_num(key_settings->JOY_FIRE4, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 8:
				get_key_name(text, options[13], key_settings->JOY_FIRE5);
				key_settings->JOY_FIRE5		= get_key_num(key_settings->JOY_FIRE5, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 9:
				get_key_name(text, options[13], key_settings->JOY_FIRE6);
				key_settings->JOY_FIRE6		= get_key_num(key_settings->JOY_FIRE6, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 10:
				get_key_name(text, options[13], key_settings->JOY_FIRE7);
				key_settings->JOY_FIRE7		= get_key_num(key_settings->JOY_FIRE7, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 11:
				get_key_name(text, options[13], key_settings->JOY_FIRE8);
				key_settings->JOY_FIRE8		= get_key_num(key_settings->JOY_FIRE8, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 12:
				get_key_name(text, options[13], key_settings->JOY_FIRE9);
				key_settings->JOY_FIRE9		= get_key_num(key_settings->JOY_FIRE9, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			case 13:
				get_key_name(text, options[13], key_settings->JOY_FIRE10);
				key_settings->JOY_FIRE10	= get_key_num(key_settings->JOY_FIRE10, ExKey==GPC_VK_RIGHT ? 1 : 0);;
				break;
			}
		}
		else if(ExKey==GPC_VK_FA){
			/* Save current settings to SMC */
			char settingsToSave[256];
			char cfgFile[128];

			GpTextOut(NULL,&gpDraw[nflip],216,25,"Saving...",254);
			
			gm_memset(settingsToSave,0,256);
			gm_sprintf(cfgFile, KEY_FILENAME, drivers[game_index].name);

			if (GpFileCreate(cfgFile, ALWAYS_CREATE, &fileHandle )==SM_OK) {
				GpFileClose( fileHandle );
				if (GpFileOpen(cfgFile, OPEN_W, &fileHandle )==SM_OK) {
					sprintf(settingsToSave, KEY_STRUCTURE, key_settings->JOY_LEFT, key_settings->JOY_RIGHT, key_settings->JOY_UP, key_settings->JOY_DOWN, key_settings->JOY_FIRE1, key_settings->JOY_FIRE2, key_settings->JOY_FIRE3, key_settings->JOY_FIRE4, key_settings->JOY_FIRE5, key_settings->JOY_FIRE6, key_settings->JOY_FIRE7, key_settings->JOY_FIRE8, key_settings->JOY_FIRE9, key_settings->JOY_FIRE10, key_settings->JOY_FIRE1_AUTO, key_settings->JOY_FIRE2_AUTO, key_settings->JOY_FIRE3_AUTO);
					GpFileWrite(fileHandle, (void *)settingsToSave, gm_lstrlen(settingsToSave)+1);
					GpFileClose( fileHandle );
					GpFATUpdate("GP://");
				}			
			}		

			GpTextOut(NULL,&gpDraw[nflip],216,25,"Saving...",0);
		}
		else if(ExKey==GPC_VK_FB){
			/* Return to options */
			clear_screen();
			draw_border();
			return;
		}
		else if(ExKey==GPC_VK_SELECT){
			/* Load default options */
			load_default_keys();
			clear_screen();
			draw_border();
		}
		else if(ExKey==GPC_VK_FR){
			switch(selected_option)
			{
			case 4: key_settings->JOY_FIRE1_AUTO=key_settings->JOY_FIRE1_AUTO?0:1; break;
			case 5: key_settings->JOY_FIRE2_AUTO=key_settings->JOY_FIRE2_AUTO?0:1; break;
			case 6: key_settings->JOY_FIRE3_AUTO=key_settings->JOY_FIRE3_AUTO?0:1; break;
			}
		}
	}
}

void adjust_sensitivity(void)
{
	int ExKey, selected_option = 0, x_Pos = 40, y_Pos = 38+30, gaps = 0;
	int x_sensitivity = 0, y_sensitivity = 0, x_reversed = 0, y_reversed = 0;
	int options_count = 4;
	unsigned long dataSize;
	F_HANDLE fileHandle;
	char filename[128];
	char text[256];
	char* options[] = 
					{	"Horz Sensitivity %d",
						"Vert Sensitivity %d",
						"Save Analogue Joystick Settings",
						"Reset Analogue Joystick Settings"};

	clear_screen();

	game_index=game_list_select(last_game_selected);

	sprintf(filename, JOY_FILENAME, drivers[game_index].name);
	if( GpFileOpen( filename, OPEN_R, &fileHandle )==SM_OK )
	{
		char* dataFromFile;
		GpFileGetSize ( filename, &dataSize );
		dataFromFile = gm_zi_malloc(dataSize);
		if( GpFileRead( fileHandle, dataFromFile, dataSize, NULL)==SM_OK )
		{
			sscanf(dataFromFile, JOY_STRUCTURE, &x_sensitivity, &y_sensitivity, &x_reversed, &y_reversed);
		}

		gm_free(dataFromFile);
		GpFileClose( fileHandle );
	}

	while(1)
	{
		nflip=(nflip?0:1);
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );

		/* Draw the background */
		draw_border();

		show_selected_game();

		gm_sprintf(text, options[0], x_sensitivity);
		gp32_gamelist_text_out(x_Pos,y_Pos,text);
		gm_sprintf(text, options[1], y_sensitivity);
		gp32_gamelist_text_out(x_Pos,y_Pos+15,text);
		strcpy(text, "Reversed ");
		strcat(text, x_reversed ? "Y" : "N");
		GpTextOut(NULL,&gpDraw[nflip],x_Pos+168,y_Pos,text,254);
		strcpy(text, "Reversed ");
		strcat(text, y_reversed ? "Y" : "N");
		GpTextOut(NULL,&gpDraw[nflip],x_Pos+168,y_Pos+15,text,254);

		gp32_gamelist_text_out(x_Pos,y_Pos+35,options[2]);
		gp32_gamelist_text_out(x_Pos,y_Pos+50,options[3]);

		gp32_gamelist_text_out(x_Pos,y_Pos+125,"Press A to select, B to return");

		/* Adjust pointer position for spacing in menu */
		gaps = selected_option > 1 ? 5 : 0;

		/* Show currently selected item */
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-10,y_Pos+gaps+(selected_option*15),">", 254 );
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-13,y_Pos+gaps+(selected_option*15),"-", 254 );

		GpSurfaceSet( &gpDraw[nflip] );
		while(GpKeyGet()!=GPC_VK_NONE) { }
		while((ExKey=GpKeyGet()) == GPC_VK_NONE) { }

		GpTextOut( NULL, &gpDraw[nflip],x_Pos-10,y_Pos+gaps+(selected_option*15),">", 0 );
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-13,y_Pos+gaps+(selected_option*15),"-", 0 );

		if(ExKey==GPC_VK_DOWN){
			selected_option++;
			selected_option = selected_option % options_count;
		}
		else if(ExKey==GPC_VK_UP){
			selected_option--;
			if(selected_option<0)
				selected_option = options_count - 1;
		}
		else if( (selected_option == 0 || selected_option == 1) &&
			((ExKey & GPC_VK_FL || ExKey & GPC_VK_LEFT) || (ExKey & GPC_VK_FR || ExKey & GPC_VK_RIGHT)) )
		{
			int i, repeat = 0;
			/* Horz/Vert Sensitivity */
			while(1)
			{
				nflip=(nflip?0:1);
				GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );
				/* Draw the background */
				draw_border();

				show_selected_game();

				if(ExKey & GPC_VK_FR || ExKey & GPC_VK_RIGHT)
				{
					if( selected_option == 0 )
					{
						x_sensitivity++;
						x_sensitivity = x_sensitivity % 256;
					}
					else if( selected_option == 1 )
					{
						y_sensitivity++;
						y_sensitivity = y_sensitivity % 256;
					}
				}		
				else if(ExKey & GPC_VK_FL || ExKey & GPC_VK_LEFT)
				{
					if( selected_option == 0 )
					{
						x_sensitivity--;
						if(x_sensitivity<0)
							x_sensitivity = 255;
					}
					else if( selected_option == 1 )
					{
						y_sensitivity--;
						if(y_sensitivity<0)
							y_sensitivity = 255;					}
				}
				gm_sprintf(text, options[0], x_sensitivity);
				gp32_gamelist_text_out(x_Pos,y_Pos,text);
				gm_sprintf(text, options[1], y_sensitivity);
				gp32_gamelist_text_out(x_Pos,y_Pos+15,text);
				strcpy(text, "Reversed ");
				strcat(text, x_reversed ? "Y" : "N");
				GpTextOut(NULL,&gpDraw[nflip],x_Pos+168,y_Pos,text,254);
				strcpy(text, "Reversed ");
				strcat(text, y_reversed ? "Y" : "N");
				GpTextOut(NULL,&gpDraw[nflip],x_Pos+168,y_Pos+15,text,254);

				gp32_gamelist_text_out(x_Pos,y_Pos+35,options[2]);
				gp32_gamelist_text_out(x_Pos,y_Pos+50,options[3]);

				gp32_gamelist_text_out(x_Pos,y_Pos+125,"Press A to select, B to return");
				GpSurfaceSet( &gpDraw[nflip] );

				if( repeat == 0 )
				{
					for(i=0;i<150;i++) {
						if (GpKeyGet() != ExKey) {
							break;
						}
						Delay(10); /* Checks every 1/100th of a second, to see if the key was relased */
					}
					if(i==150)
						repeat=1;
				}

				if (GpKeyGet() != ExKey) {
					/* End of key pressing */
					break;
				}

				Delay(100);
			}
		}
		else if(ExKey==GPC_VK_FA){
			/* Execute Selected Item */
			switch(selected_option) {
				case 0:
					/* Horz Reverse */
					strcpy(text, "Reversed ");
					strcat(text, x_reversed ? "Y" : "N");
					GpTextOut( NULL, &gpDraw[nflip],x_Pos+168,y_Pos,text, 0 );
					x_reversed = x_reversed ? 0 : 1;
					break;
				case 1:
					/* Vert Reverse */
					strcpy(text, "Reversed ");
					strcat(text, y_reversed ? "Y" : "N");
					GpTextOut( NULL, &gpDraw[nflip],x_Pos+168,y_Pos+15,text, 0 );
					y_reversed = y_reversed ? 0 : 1;
					break;
				case 2:
					/* "Save Analogue Joystick Settings" */
					GpTextOut(NULL,&gpDraw[nflip],216,25,"Saving...",254);
		
					sprintf(filename,JOY_FILENAME,drivers[game_index].name);
					if (GpFileCreate(filename, ALWAYS_CREATE, &fileHandle )==SM_OK) {
						GpFileClose(fileHandle);
						if (GpFileOpen(filename, OPEN_W, &fileHandle )==SM_OK) {
							sprintf(text, JOY_STRUCTURE, x_sensitivity, y_sensitivity, x_reversed, y_reversed);
							GpFileWrite(fileHandle, (void *)text, gm_lstrlen(text)+1);
							GpFileClose(fileHandle);
							GpFATUpdate("GP://");
						}
					}

					GpTextOut(NULL,&gpDraw[nflip],216,25,"Saving...",0);
					break;
				case 3:
					/* "Reset Analogue Joystick Settings" */
					sprintf(filename,JOY_FILENAME,drivers[game_index].name);
					confirm_delete(filename, options[selected_option]);
					break;
			}
		}
		else if(ExKey==GPC_VK_FB){
			/* Return to options */
			clear_screen();
			draw_border();
			return;
		}
	}
}

void show_options(void)
{
	int ExKey, selected_option = 0, x_Pos = 40, y_Pos = 38+30, gaps = 0;
	int options_count = 7;
	unsigned long dataSize;
	F_HANDLE fileHandle;
	char filename[128];
	char text[256];
	char* options[] = 
					{	"Delete all data for this game",
						"Use The Default Options",
						"Reset High Scores",
						"Re-assign keys for this game",
						"Adjust Joystick sensitivity",
						"Reset The Default Options",
						"Update available games list"};

	clear_screen();

	game_index=game_list_select(last_game_selected);

	while(1)
	{
		nflip=(nflip?0:1);
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );

		/* Draw the background */
		draw_border();

		/* Display the options */
		show_selected_game();

		gp32_gamelist_text_out(x_Pos,y_Pos,options[0]);
		gp32_gamelist_text_out(x_Pos,y_Pos+15,options[1]);
		gp32_gamelist_text_out(x_Pos,y_Pos+30,options[2]);

		gp32_gamelist_text_out(x_Pos,y_Pos+48,options[3]);
		gp32_gamelist_text_out(x_Pos,y_Pos+63,options[4]);

		gp32_gamelist_text_out(x_Pos,y_Pos+90,options[5]);
		gp32_gamelist_text_out(x_Pos,y_Pos+105,options[6]);

		gp32_gamelist_text_out(x_Pos,y_Pos+125,"Press A to select, B to return");

		/* Adjust pointer position for spacing in menu */
		gaps = selected_option > 2 ? 3 : 0;
		gaps += selected_option > 4 ? 12 : 0;

		/* Show currently selected item */
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-10,y_Pos+gaps+(selected_option*15),">", 254 );
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-13,y_Pos+gaps+(selected_option*15),"-", 254 );

		GpSurfaceSet( &gpDraw[nflip] );
		while(GpKeyGet()!=GPC_VK_NONE) { }
		while((ExKey=GpKeyGet()) == GPC_VK_NONE) { }

		GpTextOut( NULL, &gpDraw[nflip],x_Pos-10,y_Pos+gaps+(selected_option*15),">", 0 );
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-13,y_Pos+gaps+(selected_option*15),"-", 0 );

		if(ExKey==GPC_VK_DOWN){
			selected_option++;
			selected_option = selected_option % options_count;
		}
		else if(ExKey==GPC_VK_UP){
			selected_option--;
			if(selected_option<0)
				selected_option = options_count - 1;
		}
		else if(ExKey==GPC_VK_FA){
			/* Execute Selected Item */
			switch(selected_option) {
			case 0:
				/* "Delete ALL game data" */
				sprintf(filename,"gp:\\gpmm\\mamegp32\\roms\\%s.zip\0",drivers[game_index].name);
				if( confirm_delete(filename, options[selected_option]) )
				{
					sprintf(filename,"gp:\\gpmm\\mamegp32\\cfg\\%s.cfg\0",drivers[game_index].name);
					GpFileRemove(filename);
					GpFATUpdate("GP://");
					sprintf(filename,"gp:\\gpmm\\mamegp32\\hi\\%s.hi\0",drivers[game_index].name);
					GpFileRemove(filename);
					GpFATUpdate("GP://");
				}
				break;
			case 1:
				/* "Use Default Options" */
				sprintf(filename,"gp:\\gpmm\\mamegp32\\cfg\\%s.cfg\0",drivers[game_index].name);
				confirm_delete(filename, options[selected_option]);
				break;
			case 2:
				/* "Reset High Scores" */
				sprintf(filename,"gp:\\gpmm\\mamegp32\\hi\\%s.hi\0",drivers[game_index].name);
				confirm_delete(filename, options[selected_option]);
				break;
			case 3:
				/* Go to the sub-menu for re-configuring keys */
				configure_keys();
				break;
			case 4:
				/* Go to the sub-menu for adjust analogue control's sensitivity */
				adjust_sensitivity();
				break;
			case 5:
				/* "Reset Default Configuration" */
				confirm_delete("gp:\\gpmm\\mamegp32\\cfg\\mame.cfg\0", options[selected_option]);
				break;
			case 6:
				/* "Reload Game List" */
				update_game_list(1);
				/* Return to list of games */
				return;
			}
		}
		else if(ExKey==GPC_VK_FB){
			/* Return to list of games */
			return;
		}
	}
}

void select_game(void) {

	int repeatMode=0, ExKey=0, i, c, games_reloaded;

	/* Wait until no key pressed */
	while(GpKeyGet()!=GPC_VK_NONE) { }

	if(!gp32_fexists(gamelist_cfg)) {
		/* 1st time in, update the list of games */
		update_game_list(0);
	}

	clear_screen();

	/* Available games? */
	if(game_num_avail==0) {
		GpTextOut( NULL, &gpDraw[nflip],35,110,"ERROR: NO AVAILABLE GAMES FOUND", 254 );
		games_reloaded = 0;
		while(!games_reloaded)
		{
			while(!games_reloaded && (ExKey=GpKeyGet())!=GPC_VK_NONE) {
				/* Reset GP32 */
				if ((ExKey & GPC_VK_FL) && (ExKey & GPC_VK_FR)) {
		 			GpClockSpeedChange(67800000, 0x69032, 3);
					__asm("swi #4;");
				}
				if ((ExKey & GPC_VK_START) && (ExKey & GPC_VK_SELECT)){
					/* Update list of known games */
					update_game_list(1);
					games_reloaded = 1;
					game_list_view(&last_game_selected);
				}
			}
			while(!games_reloaded && (ExKey=GpKeyGet())==GPC_VK_NONE) {}

			/* Reset GP32 */
			if ((ExKey & GPC_VK_FL) && (ExKey & GPC_VK_FR)) {
	 			GpClockSpeedChange(67800000, 0x69032, 3);
				__asm("swi #4;");
			}
			if ((ExKey & GPC_VK_START) && (ExKey & GPC_VK_SELECT)){
				/* Update list of known games */
				update_game_list(1);
				games_reloaded = 1;
				game_list_view(&last_game_selected);
			}
		}
	} /* End of "NO AVAILABLE GAMES FOUND" section */

	/* Set Palette */
	for (c=0;c<256;c++)
		gpPalette[c]=gp32menu_Pal[c];

	/* Wait until user selects a game */
	while(1) {
		nflip=(nflip?0:1);
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );
		game_list_view(&last_game_selected);
		GpSurfaceSet( &gpDraw[nflip] );
		while((ExKey=GpKeyGet())!=GPC_VK_NONE && repeatMode==0/* && (ExKey!=LastKey)*/) {
			/* Reset GP32 */
			if ((ExKey & GPC_VK_FL) && (ExKey & GPC_VK_FR)) {
		 	 	GpClockSpeedChange(67800000, 0x69032, 3);
				__asm("swi #4;");
			}
			if ((ExKey & GPC_VK_START) && (ExKey & GPC_VK_SELECT)){
				/* Update list of known games */
				show_options();
				game_list_view(&last_game_selected);
			}
		}

		while((ExKey=GpKeyGet())==GPC_VK_NONE) {
			repeatMode = 0;	/* Reset "repeat mode" */
		}

		/* Reset GP32 */
		if ((ExKey & GPC_VK_FL) && (ExKey & GPC_VK_FR)) {
	 	 	GpClockSpeedChange(67800000, 0x69032, 3);
			__asm("swi #4;");
		}
		if ((ExKey & GPC_VK_START) && (ExKey & GPC_VK_SELECT)){
			/* Update list of known games */
			show_options();
			game_list_view(&last_game_selected);
		}

		if(repeatMode==0) {
			/* Performs the key press immediately */
			repeatMode = 1;
		}
		else if(repeatMode==1){
			/* Hold a key for 1.5 seconds and you enter "repeat mode" */
			for(i=0;i<150;i++) {
				if (GpKeyGet() != ExKey) {
					break;
				}
				Delay(10); /* Checks every 1/100th of a second, to see if the key was relased */
			}
			if(i==150) {
				/* Key held, entering full "repeat mode" */
				repeatMode=2;
			}
			else {
				repeatMode = 0;
				continue; /* This key action was already performed */
			}
		}
		else if(repeatMode==2){
			Delay(200); /* The "repeat mode" time interval */
		}
		/*LastKey=ExKey;*/

		if (ExKey & GPC_VK_UP) last_game_selected--;
		if (ExKey & GPC_VK_DOWN) last_game_selected++;
		if (ExKey & GPC_VK_FL || ExKey & GPC_VK_LEFT) last_game_selected-=11;
		if (ExKey & GPC_VK_FR || ExKey & GPC_VK_RIGHT) last_game_selected+=11;

		if (ExKey & GPC_VK_FA)
		{
			char romZip[128];
			char romDir[128];

			/* Select the game */
			game_index=game_list_select(last_game_selected);

			sprintf(romZip,"gp:\\gpmm\\mamegp32\\roms\\%s.zip\0",drivers[game_index].name);
			sprintf(romDir,"gp:\\gpmm\\mamegp32\\roms\\%s\0",drivers[game_index].name);
			if (gp32_fexists(romZip) || gp32_fexists(romDir)) {
				/* break out of the while(1) loop */
				free_memory();
				break;
			}
			else {
				int x_Pos = 40, y_Pos = 38;
				char text[128];
				clear_screen();
				draw_border();
				
				gp32_gamelist_text_out(x_Pos,y_Pos,"ERROR STARTING...");
				gp32_gamelist_text_out(x_Pos,y_Pos+15,drivers[game_index].description);
				sprintf(text, "Could not find '%s'", drivers[game_index].name);
				gp32_gamelist_text_out(x_Pos,y_Pos+60,text);
				gp32_gamelist_text_out(x_Pos,y_Pos+75,"in directory...");
				gp32_gamelist_text_out(x_Pos,y_Pos+90,"gp:\\gpmm\\mamegp32\\roms");
				gp32_gamelist_text_out(x_Pos,y_Pos+150,"Press any key to return to list");

				while(GpKeyGet()!=GPC_VK_NONE) { }
				while(GpKeyGet()==GPC_VK_NONE) { }
			}
		}
	}

	clear_screen();
}

